package topdown.term;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.stream.Collectors;

public class FunctionalSymbol implements Term {
    /* name of the function - functional symbol */
    final String symbol;
    /* arguments of the function - terms */
    final List<Term> arguments;
    /* arity of the function - can be 0 in case of a constant */
    final int arity;
    
    public FunctionalSymbol(String symbol, List<Term> arguments) {
        this.symbol = symbol;
        this.arguments = arguments;
        this.arity = arguments.size();
    }

    public FunctionalSymbol(String symbol) {
        this.symbol = symbol;
        this.arguments = new ArrayList<>();
        this.arity = 0;
    }
    
    @Override
    public String toString() {
        return symbol
                + "("
                + arguments.stream()
                        .map(Object::toString)
                        .collect(Collectors.joining(","))
                + ")";
    }
    
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof FunctionalSymbol) {
            FunctionalSymbol f = (FunctionalSymbol) obj;
            if (!symbol.equals(f.symbol)) return false;
            if (arity != f.arity) return false;
            for (int i=0; i<arity; i++) {
                if (!arguments.get(i).equals(f.arguments.get(i))) {
                    return false;
                }
            }
            return true;
        }
        return false;
    }
    
    @Override
    public int hashCode() {
        return (symbol + "f" + arity).hashCode();
    }
    
    @Override
    public Term substitute(Map<Variable, Term> substitution) {
        List<Term> arg = arguments.stream()
                .map(x -> x.substitute(substitution))
                .collect(Collectors.toList());
        return new FunctionalSymbol(symbol, arg);
    }
    
    @Override
    public Term copy() {
        List<Term> arg = arguments.stream()
                .map(x -> x.copy())
                .collect(Collectors.toList());
        return new FunctionalSymbol(symbol, arg);
    }

}
